# Vertec vtcplanning typestub module
#
# used instead of the builtin module for static analysis in
# linter enabled editors (eg VSCode).
#
# 2022 Copyright Vertec AG, Weststrasse 75, CH-8003 Zürich

from collections.abc import Callable
import vtcapp
import vtcplanningcore
from datetime import date


# region generic planning lists


class ControllerBase(object):
    """
    Base class for all List Controllers for resource planning tables.
    Contains base functionality.

    This is an abstract List Controller and can not be used directly.
    """

    context: object = ...
    """
    Context object of List Controller. Can be of type 'Container' or 'UserEintrag'.
    """

    owner: object = ...
    """
    Owner object of List Controller. Can be of type 'LinkRolle', 'ViewType' or 'AbfrageOrdner'.
    """

    expected_context_type: str = ...
    """
    Template attribute can be set in subclasses.
    If this attribute is set, the method 'self.get_context' will validated
    the List Controller context type against that expected context type.
    """

    is_readonly: bool = ...
    """
    Template attribute can be set in subclasses.
    Set this attribute in a subclass to indicate that a list shall be readonly or not.
    """

    decrement_provider_start_date_by_interval_count: int = ...
    """
    Template attribut that can be set to load additional planning intervals before the
    planning period start date into the RessoucePlanningProvider.
    Indicates how many intervals shall be loaded additionaly before to the planning period start date.
    Overwrite this attribute to change how many additional intervals shall be loaded before the planning period start date.
    """

    increment_provider_end_date_by_interval_count: int = ...
    """
    Template attribut that can be set to load additional planning intervals after the
    planning period end date into the RessoucePlanningProvider.
    Indicates how many intervals shall be loaded additionaly after to the planning period end date.
    Overwrite this attribute to change how many additional intervals shall be loaded after the planning period end date.
    """

    def get_dynamic_column_count(self, subscriber: object) -> int:
        """
        Standard method that is called to get the count of 
        dynamic column groups for this list.
        """
        ...

    def get_dynamic_column_title(self, column_index: int, subscriber: object) -> str:
        """
        Standard method that is called to get the title of a
        dynamic column group for this list.
        """
        ...

    def get_context(self) -> object:
        """
        Getter method for 'self.context' attribute, ensures context object has been set.
        If attribut 'self.expected_context_type' is set, this method will throw an exception
        if the context is of wrong type.
        """
        ...

    def ensure_correct_context_type(self) -> None:
        """
        If attribut 'self.expected_context_type' is set, this method will throw an exception
        if the context is of wrong type.
        """
        ...

    def ensure_is_writable(self, message: str = None) -> None:
        """
        Method to ensure, that List Controller is NOT readonly.
        If List Controller is readonly, an exception will be thrown.
        This methode uses 'self.is_readonly' to evaluate, if a List Controller is readonly or not.
        """
        ...

    def get_target_classname(self) -> str:
        """
        Get TargetClass from ViewType related to this List Controllers.
        """
        ...

    def get_planning_level(self) -> str:
        """
        Get planning level.
        Planning level can be 'Projekt' or 'ProjektPhase'.
        Method ensures that planning level has been loaded from 'vtcplanningcore.get_planning_level'.
        """
        ...

    def get_planning_interval_type(self) -> str:
        """
        Gets the planning interval type.
        Indicates if planning interval type is 'day', 'week' or 'month'.
        Method ensures that planning interval type has been loaded from 'vtcplanningcore.get_interval_type'.
        """
        ...

    def get_provider(self) -> vtcplanningcore.ResourcePlanningProvider:
        """
        Getter method for private 'self._provider' attribute, ensures that ResourcePlanningProvider has been initialized.
        """
        ...

    def get_provider_start_date(self, start_date: date) -> date:
        """
        Gets the correct start date for the RessoucePlanningProvider.
        If attribute 'self.decrement_provider_start_date_by_interval_count' is
        set, the start date will be decremented by 'self.decrement_provider_start_date_by_interval_count'.
        """
        ...

    def get_provider_end_date(self, end_date: date) -> date:
        """
        Gets the correct end date for the RessoucePlanningProvider.
        If attribute 'self.decrement_provider_start_date_by_interval_count' is
        set, the start date will be decremented by 'self.decrement_provider_start_date_by_interval_count'.
        """
        ...

    def initialize_with_period(self, start: date, end: date, subscriber: object) -> None:
        """
        Template method that must be implemented in subclasses.
        Method will be call during initialization of a resource planning List Controller.
        """
        ...

    def setup_provider(self, subscriber: object) -> None:
        """
        Template method that must be implemented in subclasses.
        Helper method to setup the ResourcePlanningProvider.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def create_and_set_provider(self,
                                source_entries: object | list[object],
                                start_date: date,
                                end_date: date,
                                subscriber: object) -> None:
        """
        Creates new instance of ResourcePlanningProvider and sets the private 'self._provider' attribute.
        Calculates the start and end date for the provider.
        """
        ...

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Template function that must be implemented in subclasses.
        Standard method that returns the row objects for the table.
        """
        ...

    def get_capacity_method_arguments(self,
                                      column_index: int,
                                      rowobj: object | None,
                                      subscriber: object | None) -> tuple[object, date, date | None]:
        """
        Template function that must be implemented in subclasses if capacity calculation methods are used.
        Helper method to prepare the capacity method arguments.
        Capacity method arguments must be returned as tuple of type (worker, date_from, date_to).
        Thereof 'worker' and 'date_from' can not be None.
        """
        ...

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Template function that must be implemented in subclasses.
        Standard method to get the plannde minutes for a
        column index and a row object.
        """
        ...

    def set_planned_minutes(self, column_index: int, rowobj: object, value: int | None) -> None:
        """
        Template function that must be implemented in subclasses which are NOT readonly.
        Standard method to set the plannded minutes for a
        column index and a row object.
        """
        ...

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Template function that must be implemented in subclasses.
        Standard method to get the total planned minutes for
        a row object.
        """
        ...

    def generic_get_capacity(self,
                             get_capacity_method: Callable[[object, date, date, object], int],
                             column_index: int | None,
                             rowobj: object | None,
                             subscriber: object | None) -> int:
        """
        Helper method to get the capacity of a worker. Specify with get_capacity_method
        the capacity method (gross, net, remaining). If column_index is set to None, the
        capacity will be calculated for the whole planning period.
        """
        ...

    def get_net_capacity(self, column_index: int | None, rowobj: object, subscriber: object) -> int:
        """
        Standard method to get the net capacity of a worker based on a
        column index and row object.
        If column index is None, capacity will be calculated for row object only.
        """
        ...

    def get_gross_capacity(self, column_index: int | None, rowobj: object, subscriber: object) -> int:
        """
        Standard method to get the gross capacity of a worker based on a
        column index and row object.
        If column index is None, capacity will be calculated for row object only.
        """
        ...

    def get_custom_net_capacity(self, column_index: int | None, rowobj: object, subscriber: object) -> int:
        """
        Standard method to get the custom net capacity of a worker based on a
        column index and row object.
        If column index is None, capacity will be calculated for row object only.
        """
        ...

    def get_remaining_capacity(self, column_index: int | None, rowobj: object, subscriber: object) -> int:
        """
        Standard method to get the remaining capacity of a worker base on a
        column index and row object.
        If column index is None, capacity will be calculated for row object only.
        """
        ...

    def get_custom_remaining_capacity(self, column_index: int | None, rowobj: object, subscriber: object) -> int:
        """
        Standard method to get the remaining capacity of a worker base on a
        column index and row object.
        If column index is None, capacity will be calculated for row object only.
        """
        ...

    def get_is_cell_readonly(self, column_index: int, rowobj: object, subscriber: object) -> bool:
        """
        Standard method to check if a given cell is readonly.
        This implementation just checks, if List Controller is readonly or not.
        """
        ...


class TimeTableControllerBase(ControllerBase):
    """
    Base class for all List Controllers for time tables used in our ressource planning.
    Contains base functionality.

    This is an abstract List Controller and can not be used directly.
    """

    period_start: date = ...
    """
    Start date of planning period.
    Should be set during 'self.initialize_with_period'.
    """

    period_end: date = ...
    """
    End date of planning period.
    Should be set during 'self.initialize_with_period'.
    """

    def initialize_with_period(self, start: date, end: date, subscriber: object) -> None:
        """
        Standard method that will be called whenever the planning list
        is initialized or the chosen period is changed.
        """
        self.period_start = start
        self.period_end = end
        self.setup_provider(subscriber)

    def get_date_range(self) -> list[date]:
        """
        Get date range based on 'self.period_start' and 'self.period_end'.
        Method ensures, that data range has been correctly calculated.
        """
        ...

    def get_date_by_column_index(self, column_index: int) -> date:
        """
        Get date from date range by column index.
        """
        ...

    def get_dynamic_column_count(self, subscriber: object) -> int:
        """
        Standard method that is called to get the count of 
        dynamic column groups in the planning list (e.g. count of intervals).
        """
        ...

    def get_dynamic_column_title(self, column_index: int, subscriber: object) -> str:
        """
        Standard method that is called to get the title of a
        dynamic column group (interval).
        """
        ...


class SingleObjectTimeTableController(TimeTableControllerBase):
    """
    List Controller for time tables on single objects used in our resource planning.

    This List Controller will load one interval additionally before the
    planning period to have the tables prepopulated with rows, which contains
    planning values in the previous interval.
    """

    decrement_provider_start_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly before to the planning period start date.
    Overwrite this attribute to change how many additional intervals shall be loaded before the planning period start date.
    Default value: 1 interval
    
    We initialize the provider with one date interval before the
    specified start to display entries that have planning data
    in the interval before our chosen period.
    """

    expected_context_type: str = ...
    """
    This List Controller is only to be expected to be used on single objects, not on containers.
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for SingleObjectTimeTableController.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the list.
        """
        ...

    def get_row_objects_type(self) -> str:
        """
        Standard method that returns the type of the row objects for the pivot table.
        """
        ...

    def add_row_object(self, obj: object) -> None:
        """
        Standard method that is called, when a new row
        is added to the planning list.
        """
        ...

    def get_source_entry(self) -> object:
        """
        Get source entry from List Controller Context.
        Ensures that context is of the expected type.
        """
        ...

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and a row object.
        """
        ...

    def set_planned_minutes(self, column_index: int, rowobj: object, value: int | None) -> None:
        """
        Standard method to set the plannded minutes for a
        column index and a row object.
        """
        ...

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        """
        ...

    def get_is_cell_readonly(self, column_index: int, rowobj: object, subscriber: object) -> bool:
        """
        Standard method to check if a given cell is readonly.
        This implementation evaluates write access for cell if 'self.is_readonly' is True.
        """
        ...

    def get_capacity_method_arguments(self,
                                      column_index: int | None,
                                      rowobj: object,
                                      subscriber: object) -> tuple[object, date, date | None]:
        """
        Helper method to prepare the capacity method arguments for SingleObjectTimeTableController.
        """
        ...


class ReadonlySingleObjectTimeTableController(SingleObjectTimeTableController):
    """
    List Controller for time tables on single objects used in our resource planning.

    This List Controller resolves automatically the projects and phases objects to
    the correct planning level set in the system settings and will aggregate the
    planning values accordingly. Therefore, the table will be readonly.
    """

    is_readonly: bool = ...
    """
    Indicates that table will be readonly.
    """

    decrement_provider_start_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly before to the planning period start date.
    Overwrite this attribute to change how many additional intervals shall be loaded before the planning period start date.
    Default value: 0 interval
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for SingleObjectTimeTableController.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def get_source_entry(self) -> object:
        """
        Get source entry from List Controller Context.
        Ensures that context is of the expected type.
        """
        ...

    def get_source_entries(self, subscriber: object = None) -> list[object]:
        """
        Get source entries from List Controller Context
        and converts the to the correct planning level.
        Ensures that context is of the expected type.
        """
        ...

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and a row object.
        """
        ...

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        """
        ...

    def get_capacity_method_arguments(self,
                                      column_index: int,
                                      rowobj: object,
                                      subscriber: object) -> tuple[object, date, date | None]:
        """
        Helper method to prepare the capacity method arguments for SingleObjectTimeTableController.
        """
        if not rowobj:
            raise ValueError("Can not get capacity if row object is None")

        if is_abstractworker(rowobj):
            worker = rowobj
        else:
            source_entries = self.get_source_entries(subscriber)
            worker = get_first_object(
                source_entries)

        if column_index is None:
            date_from = self.period_start
            date_to = self.period_end
        else:
            date_from = self.get_date_by_column_index(column_index)
            date_to = None

        return (worker, date_from, date_to)


class ReadonlyOtherSideSingleObjectTimeTableController(ReadonlySingleObjectTimeTableController):
    """
    List Controller for time tables on single objects used in our resource planning.
    This table will show single object (or the childs of the single object) as rows
    instead of the otherside objects.

    This List Controller resolves automatically the projects and phases objects to
    the correct planning level set in the system settings.

    This List Controller will load one interval additionally before the
    planning period to have the tables prepopulated with rows, which contains
    planning values in the previous interval.
    """

    decrement_provider_start_date_by_interval_count = 1
    """
    Indicates how many intervals shall be loaded additionaly before to the planning period start date.
    Overwrite this attribute to change how many additional intervals shall be loaded before the planning period start date.
    Default value: 1 interval
    
    We initialize the provider with one date interval before the
    specified start to display entries that have planning data
    in the interval before our chosen period.
    """

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the list.
        """
        return self.get_source_entries(subscriber)

    def get_row_objects_type(self) -> str:
        """
        Standard method that returns the type of the row objects for the pivot table.
        """
        if self.get_target_classname() == "AbstractWorker":
            return "AbstractWorker"

        return self.get_planning_level()

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and a row object.
        """
        date = self.get_date_by_column_index(column_index)
        source_entries = resolve_projects_or_phases(
            rowobj, self.get_planning_level(), subscriber)

        return self.get_provider().get_planned_minutes_aggregated(
            source_entries, None, date, None, subscriber)

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        """
        source_entries = resolve_projects_or_phases(
            rowobj, self.get_planning_level(), subscriber)

        return self.get_provider().get_planned_minutes_aggregated(
            source_entries, None, self.period_start, self.period_end, subscriber)


class ReadonlyContainerTimeTableController(TimeTableControllerBase):
    """
    List Controller for time tables on object containers used in our resource planning.

    This List Controller resolves automatically the projects and phases objects to
    the correct planning level set in the system settings and will aggregate the
    planning values accordingly. Therefore, the table will be readonly.
    """

    is_readonly = True
    """
    Indicates that table will be readonly.
    """

    expected_context_type = "Container"
    """
    This List Controller is only to be expected to be used on containers, not on single objects.
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for ReadonlyContainerTimeTableController.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        source_entries = self.get_source_entries(subscriber)
        self.create_and_set_provider(
            source_entries, self.period_start, self.period_end, subscriber)

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the list.
        """
        source_entries = self.get_source_entries(subscriber)
        return resolve_projects_or_phases(source_entries, self.get_target_classname(), subscriber)

    def get_source_entries(self, subscriber: object = None) -> list[object]:
        """
        Get source entries from List Controller Context
        and converts the to the correct planning level.
        Ensures that context is of the expected type.
        """
        if not hasattr(self, "_source_entries"):
            self.ensure_correct_context_type()
            context_entries = self.get_context_entries()
            self._source_entries = resolve_projects_or_phases(
                context_entries, self.get_planning_level(), subscriber)

        return self._source_entries

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and row object.
        """
        date = self.get_date_by_column_index(column_index)
        entries = resolve_projects_or_phases(
            rowobj, self.get_planning_level(), subscriber)

        return self.get_provider().get_planned_minutes_aggregated(
            entries, None, date, None, subscriber)

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        """
        entries = resolve_projects_or_phases(
            rowobj, self.get_planning_level(), subscriber)

        return self.get_provider().get_planned_minutes_aggregated(
            entries, None, self.period_start, self.period_end, subscriber)

    def get_capacity_method_arguments(self,
                                      column_index: int | None,
                                      rowobj: object,
                                      subscriber: object) -> tuple[object, date, date | None]:
        """
        Helper method to prepare the capacity method arguments for ReadonlyContainerTimeTableController.
        """
        ...


class ReadonlyOtherSideContainerTimeTableController(ReadonlyContainerTimeTableController):
    """
    List Controller for time tables on object containers used in our resource planning.
    This table will show the container objects as rows instead of the otherside objects.

    This List Controller resolves automatically the projects and phases objects to
    the correct planning level set in the system settings and will aggregate the
    planning values accordingly. Therefore, the table will be readonly.
    """

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the list.
        """
        ...

    def get_row_objects_type(self) -> str:
        """
        Standard method that returns the type of the row objects for the pivot table.
        """
        if self.get_target_classname() == "AbstractWorker":
            return self.get_planning_level()

        return "AbstractWorker"

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and row object.
        """
        date = self.get_date_by_column_index(column_index)
        return self.get_provider().get_planned_minutes_aggregated(None, rowobj, date, None, subscriber)

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        """
        return self.get_provider().get_planned_minutes_aggregated(None, rowobj, self.period_start, self.period_end, subscriber)

    def get_capacity_method_arguments(self,
                                      column_index: int,
                                      rowobj: object,
                                      subscriber: object) -> tuple[object, date, date | None]:
        """
        Helper method to prepare the capacity method arguments for ReadonlyContainerTimeTableController.
        """
        if not rowobj:
            raise ValueError("Can not get capacity if row object is None")

        if is_abstractworker(rowobj):
            worker = rowobj
        else:
            raise ValueError(
                "Can not get capacity if row object is not of type 'AbstractWorker'")

        if column_index is None:
            date_from = self.period_start
            date_to = self.period_end
        else:
            date_from = self.get_date_by_column_index(column_index)
            date_to = None

        return (worker, date_from, date_to)


class PivotTableControllerBase(ControllerBase):
    """
    Base class for all List Controllers for pivot tables used in our ressource planning.
    Contains base functionality.

    This is an abstract List Controller and can not be used directly.
    """

    interval_start = None
    """
    Start date of planning interval of pivot table.
    Should be set during 'self.initialize_with_period'.
    """

    expected_context_type = "Container"
    """
    Pivot Table Controllers are only to be expected to be used on containers, not on single objects.
    """

    def initialize_with_period(self, start: date, end: date, subscriber: object) -> None:
        """
        Standard method that will be called whenever the planning list
        is initialized or the chosen period is changed.
        """
        self.interval_start = start
        self.setup_provider(subscriber)

    def get_source_entries(self, subscriber: object = None) -> list[object]:
        """
        Get source entries from List Controller Context..
        Ensures that context is of the expected type.
        """
        if not hasattr(self, "_source_entries"):
            self.ensure_correct_context_type()
            self._source_entries = self.get_context_entries()

        return self._source_entries

    def get_row_objects_type(self) -> str:
        """
        Template method that must be implemented in subclasses.
        Standard method that returns the type of the row objects for the pivot table.
        """
        ...

    def get_column_objects(self, subscriber: object = None) -> list[object]:
        """
        Template method that must be implemented in subclasses.
        Standard method that returns the column objects for the pivot table.
        """
        ...

    def get_column_object_by_column_index(self, column_index: int, subscriber: object = None) -> object:
        """
        Get column object by column index.
        """
        ...

    def get_dynamic_column_count(self, subscriber: object) -> int:
        """
        Standard method that is called to get the count of 
        dynamic column groups in the planning table (e.g. count of column objects).
        """
        ...

    def get_dynamic_column_title(self, column_index: int, subscriber: object) -> str:
        """
        Standard method that is called to get the title of a
        dynamic column group (string representation of column object).
        """
        ...

    def get_capacity_method_arguments(self,
                                      column_index: int | None,
                                      rowobj: object,
                                      subscriber: object) -> tuple[object, date, date | None]:
        """
        Helper method to prepare the capacity method arguments for PivotTableControllerBase and super classes.
        """
        ...


class RegularPivotTableController(PivotTableControllerBase):
    """
    List Controller for pivot tables on object containers used in our resource planning.
    Regular pivot tables show the container objects as rows and the otherside objects
    as transient columns.

    This List Controller will load one interval additionally before and after the
    planning interval of the pivot table to have the table prepopulated with rows,
    which contains planning values in the previous or subsequent interval.
    """

    decrement_provider_start_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly before to the planning interval.
    Overwrite this attribute to change how many additional intervals shall beloaded before the planning interval.
    Default value: 1 interval
    
    We initialize the provider with one date interval before the
    specified interval to display entries that have planning data
    in the interval before our chosen interval.
    """

    increment_provider_end_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly after to the planning planning interval.
    Overwrite this attribute to change how many additional intervals shall be loaded after the planning interval.
    Default value: 1 interval
    
    We initialize the provider with one date interval after the
    specified interval to display entries that have planning data
    in the interval after our chosen interval.
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for RegularPivotTableController.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def get_row_objects(self, subscriber: object = None) -> list[object]:
        """
        Standard method that returns the row objects for the regular pivot table.
        """
        ...

    def get_row_objects_type(self) -> str:
        """
        Standard method that returns the type of the row objects for the pivot table.
        """
        ...

    def get_column_objects(self, subscriber: object = None) -> list[object]:
        """
        Standard method that returns the column objects for the regular pivot table.
        """
        ...

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and row object.
        """
        ...

    def set_planned_minutes(self, column_index: int, rowobj: object, value: int | None) -> None:
        """
        Standard method to set the plannded minutes for a
        column index and a row object.
        """
        ...

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        """
        ...


class ReadonlyRegularPivotTableController(RegularPivotTableController):
    """
    List Controller for pivot tables on object containers used in our resource planning.
    Regular pivot tables show the container objects as rows and the otherside objects
    as transient columns.

    This List Controller resolves automatically the projects and phases objects to
    the correct planning level set in the system settings and will aggregate the
    planning values accordingly. Therefore, the table will be readonly.
    """

    decrement_provider_start_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly before to the planning interval.
    Overwrite this attribute to change how many additional intervals shall be loaded before the planning interval.
    Default value: 0 interval
    """

    increment_provider_end_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly after to the planning planning interval.
    Overwrite this attribute to change how many additional intervals shall be loaded after the planning interval.
    Default value: 0 interval
    """

    is_readonly: bool = ...
    """
    Indicates that table will be readonly.
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for ReadonlyRegularPivotTableController.
        This method automatically resolves discrepancies between target class and planning level.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def get_source_entries(self, subscriber: object = None) -> list[object]:
        """
        Get source entries from List Controller Context
        and converts them to the correct planning level.
        Ensures that context is of the expected type.
        """
        ...

    def get_row_objects(self, subscriber: object = None) -> list[object]:
        """
        Standard method that returns the row objects for the regular pivot table.
        This method automatically resolves discrepancies between target class and planning level.
        """
        ...

    def get_column_objects(self, subscriber: object = None) -> list[object]:
        """
        Standard method that returns the column objects for the regular pivot table.
        This method automatically resolves discrepancies between target class and planning level.
        """
        ...

    def get_row_objects_type(self) -> str:
        """
        Standard method that returns the type of the row objects for the pivot table.
        """
        ...

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and row object.
        This method automatically resolves discrepancies between target class and planning level.
        """
        ...

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        This method automatically resolves discrepancies between target class and planning level.
        """
        ...


class RegularSingleObjectPivotTableController(RegularPivotTableController):
    """
    List Controller for pivot tables on single objects used in our resource planning.
    Regular pivot tables show the single object (or the childs of the single object) as rows
    and the otherside objects as transient columns.

    This List Controller resolves automatically the projects and phases objects to
    the correct planning level set in the system settings.

    This List Controller will load one interval additionally before and after the
    planning interval of the pivot table to have the table prepopulated with rows,
    which contains planning values in the previous or subsequent interval.
    """

    expected_context_type: str = ...
    """
    Single Object Pivot Table Controllers are only to be expected to be used on single objects, not on containers.
    """

    def get_source_entries(self, subscriber: object = None) -> list[object]:
        """
        Get source entries from List Controller Context..
        Ensures that context is of the expected type.
        """
        ...


class MirroredPivotTableController(PivotTableControllerBase):
    """
    List Controller for pivot tables on object containers used in our resource planning.
    Mirrored pivot tables show the container objects as columns and the otherside objects
    as transient rows. Mirrored pivot tables do not support ghost rows.

    This List Controller will load one interval additionally before and after the
    planning interval of the pivot table to have the table prepopulated with rows,
    which contains planning values in the previous or subsequent interval.
    """

    decrement_provider_start_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly before to the planning interval.
    Overwrite this attribute to change how many additional intervals shall beloaded before the planning interval.
    Default value: 1 interval
    
    We initialize the provider with one date interval before the
    specified interval to display entries that have planning data
    in the interval before our chosen interval.
    """

    increment_provider_end_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly after to the planning planning interval.
    Overwrite this attribute to change how many additional intervals shall be loaded after the planning interval.
    Default value: 1 interval
    
    We initialize the provider with one date interval after the
    specified interval to display entries that have planning data
    in the interval after our chosen interval.
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for MirroredPivotTableController.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the mirrored pivot table.
        """
        ...

    def get_row_objects_type(self) -> str:
        """
        Standard method that returns the type of the row objects for the pivot table.
        """
        ...

    def add_row_object(self, obj: object) -> None:
        """
        Standard method that is called, when a new row
        is added to the planning list.
        """
        ...

    def get_column_objects(self, subscriber: object = None) -> list[object]:
        """
        Standard method that returns the column objects for the regular pivot table.
        """
        ...

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and row object.
        """
        ...

    def set_planned_minutes(self, column_index: int, rowobj: object, value: int | None) -> None:
        """
        Standard method to set the plannded minutes for a
        column index and a row object.
        """
        ...

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        """
        ...


class ReadonlyMirroredPivotTableController(MirroredPivotTableController):
    """
    List Controller for pivot tables on object containers used in our resource planning.
    Mirrored pivot tables show the container objects as columns and the otherside objects
    as transient rows. Mirrored pivot tables do not support ghost rows.

    This List Controller resolves automatically the projects and phases objects to
    the correct planning level set in the system settings and will aggregate the
    planning values accordingly. Therefore, the table will be readonly.
    """

    decrement_provider_start_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly before to the planning interval.
    Overwrite this attribute to change how many additional intervals shall be loaded before the planning interval.
    Default value: 0 interval
    """

    increment_provider_end_date_by_interval_count: int = ...
    """
    Indicates how many intervals shall be loaded additionaly after to the planning planning interval.
    Overwrite this attribute to change how many additional intervals shall be loaded after the planning interval.
    Default value: 0 interval
    """

    is_readonly: bool = ...
    """
    Indicates that table will be readonly.
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for ReadonlyMirroredPivotTableController.
        This method automatically resolves discrepancies between target class and planning level.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def get_source_entries(self, subscriber: object = None) -> list[object]:
        """
        Get source entries from List Controller Context
        and converts them to the correct planning level.
        Ensures that context is of the expected type.
        """
        ...

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the readonly mirrored pivot table.
        This method automatically resolves discrepancies between target class and planning level.
        """
        ...

    def get_column_objects(self, subscriber: object = None) -> list[object]:
        """
        Standard method that returns the column objects for the regular pivot table.
        This method automatically resolves discrepancies between target class and planning level.
        """
        ...

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and row object.
        This method automatically resolves discrepancies between target class and planning level.
        """
        ...

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        This method automatically resolves discrepancies between target class and planning level.
        """
        ...


class MirroredSingleObjectPivotTableController(MirroredPivotTableController):
    """
    List Controller for pivot tables on single objects used in our resource planning.
    Mirrored pivot tables show the single object (or the childs of the single object)
    as columns and the otherside objects as transient rows.
    Mirrored pivot tables do not support ghost rows.

    This List Controller resolves automatically the projects and phases objects to
    the correct planning level set in the system settings.

    This List Controller will load one interval additionally before and after the
    planning interval of the pivot table to have the table prepopulated with rows,
    which contains planning values in the previous or subsequent interval.
    """

    expected_context_type: str = ...
    """
    Single Object Pivot Table Controllers are only to be expected to be used on single objects, not on containers.
    """

    def get_source_entries(self, subscriber: object = None) -> list[object]:
        """
        Get source entries from List Controller Context..
        Ensures that context is of the expected type.
        """
        ...


# endregion

# region planning tables for planning on phases


class ReadonlyPhasesContainerTimeTableController(ReadonlyContainerTimeTableController):
    """
    List Controller for time tables on object containers used in our resource planning.

    This is a special List Controller which can only be used while planning on phases
    and can only be used on projects and abstract workers containers.

    Table will show the related phases of the projects as transient rows. Table will be readonly.
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for ReadonlyContainerTimeTableController.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the list.
        """
        ...

    def get_source_entries(self, subscriber: object = None) -> list[object]:
        """
        Get source entries from List Controller Context
        and converts the to the correct planning level.
        Ensures that context is of the expected type.
        """
        ...

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the list.
        """
        ...

    def get_planned_minutes(self, column_index: int, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the plannde minutes for a
        column index and row object.
        """
        ...

    def get_planned_minutes_aggregated(self, rowobj: object, subscriber: object) -> int | None:
        """
        Standard method to get the total planned minutes for
        a row object.
        """
        ...

    def get_capacity_method_arguments(self,
                                      column_index: int,
                                      rowobj: object,
                                      subscriber: object) -> tuple[object, date, date | None]:
        """
        Helper method to prepare the capacity method arguments for ReadonlyContainerTimeTableController.
        """
        ...


class ReadonlyProjectsSingleObjectTimeTableController(ReadonlySingleObjectTimeTableController):
    """
    List Controller for time tables on single objects used in our resource planning.

    This is a special List Controller which can only be used while planning on phases
    and can only be used on a single abstract worker object.

    Table will show the related projects of the phases with plan values as transient rows.
    Table will be readonly, because plan values will be aggregated to projects.
    """

    expected_context_type: str = ...
    """
    This List Controller is only to be expected to be used on single abstract worker objects, not on containers.
    """

    def setup_provider(self, subscriber: object) -> None:
        """
        Setup of the ResourcePlanningProvider for ReadonlyContainerTimeTableController.
        After the setup, the ResourcePlanningProvider can be used via function 'self.get_provider'.
        """
        ...

    def get_row_objects(self, subscriber: object) -> list[object]:
        """
        Standard method that returns the row objects for the list.
        """
        ...

    def get_row_objects_type(self) -> str:
        """
        Standard method that returns the type of the row objects for the pivot table.
        """
        ...

    def get_source_entry(self) -> object:
        """
        Get source entry from List Controller Context.
        Ensures that context is of the expected type.
        """
        ...


# endregion

# region Business Object Helper module functions


def is_of_type(
        obj_or_obj_list: object | list[object],
        expected_type: str,
        evaluate_full_list: bool = False) -> bool:
    """
    Helper method to test, if the object or list of objects is of the expected type.
    In case of a list, only the first object will be evaluated if argument 'evaluate_full_list = False'.
    If None or in case of an exception, False will be returned.
    """
    ...


def is_abstractworker(obj_or_obj_list: object | list[object]) -> bool:
    """
    Helper method to test, if the object or list of objects is of type 'AbstracWorker'.
    In case of a list, only the first object will be evaluated.
    If None or in case of an exception, False will be returned.
    """
    ...


def is_projectworker(obj_or_obj_list: object | list[object]) -> bool:
    """
    Helper method to test, if the object or list of objects is of type 'Projektbearbeiter'.
    In case of a list, all objects will be evaluated.
    If None or in case of an exception, False will be returned.
    """
    ...


def is_planningworker(obj_or_obj_list: object | list[object]) -> bool:
    """
    Helper method to test, if the object or list of objects is of type 'PlanningWorker'.
    In case of a list, all objects will be evaluated.
    If None or in case of an exception, False will be returned.
    """
    ...


def is_project(obj_or_obj_list: object | list[object]) -> bool:
    """
    Helper method to test, if the object or list of objects is of type 'Projekt'.
    In case of a list, only the first object will be evaluated.
    If None or in case of an exception, False will be returned.
    """
    ...


def is_phase(obj_or_obj_list: object | list[object]) -> bool:
    """
    Helper method to test, if the object or list of objects is of type 'ProjektPhase'.
    In case of a list, only the first object will be evaluated.
    If None or in case of an exception, False will be returned.
    """
    ...


def get_phases_for_projects(
        project_or_projects: object | list[object],
        subscriber: object | None = None) -> list[object]:
    """
    Helper method to get all phases from a project or a list of projects.
    """
    ...


def get_projects_for_phases(phase_or_phases: object | list[object]) -> list[object]:
    """
    Helper method to get all projects from a phase or a list of phases.
    """
    ...


def get_as_list(obj_or_obj_list: object | list[object] | None) -> list[object]:
    """
    Wraps object into list if it is not allready an object list.
    If None, an empty list will be returned.
    """
    ...


def get_first_object(obj_or_obj_list: object | list[object]) -> object:
    """
    If is list, return first object, else returns object itself.
    Throws exception in case of None.
    """
    ...


def is_single_object(obj_or_obj_list: object | list[object] | None) -> bool:
    """
    Helper method to check, if 'obj_or_obj_list' is a single object.
    Returns False if 'obj_or_obj_list' is None.
    """
    ...


def resolve_projects_or_phases(
        entries: object | list[object] | None,
        desired_class: str,
        subscriber: object | None = None) -> list[object] | None:
    """
    This resolves entries to the desired class type.
    This function can resolve phases to projects or project to phases.
    In all other cases, it will just return the entries.
    """
    ...

# endregion

# region renderer


class RendererBase(object):
    """
    Base class for all renderers used for resource planning tables.
    Contains base functionality to ensure that the List Controller
    is a resource planning table controller.
    This is an abstract List Controller and can not be used directly.
    """

    context: object = ...
    """
    Context object of List Controller. Can be of type 'Ordner' or 'LinkContainer'.
    """

    controller: ControllerBase = ...
    """
    List Controller object of Custom Renderer. Is None if Custom Rendere is used in a list without a List Controller.
    Use the getter method 'self.get_controller' to savely access the controller instance.
    """

    def get_context(self) -> object:
        """
        Getter method for 'self.context' attribute, ensures context object has been set.
        Context object of List Controller. Can be of type 'Ordner' or 'LinkContainer'.
        """
        ...

    def get_controller(self) -> ControllerBase:
        """
        Getter method for 'self.controller' attribute, ensures List Controller object has been set and that it is a planning table controller.
        """
        ...


class MinutesRendererBase(RendererBase):
    """
    Base class for all renderers used for resource planning tables.
    Contains base functionality to ensure that the List Controller
    is a resource planning table controller.
    This is an abstract List Controller and can not be used directly.
    """

    converter: str = ...
    """
    Converter to convert values into minutes.
    Standard attribute to set the converter.
    """

    value_type: str = ...
    """
    Ensures that value type will be integer.
    """

    def get_column_index_from_expression(self, expression: str | None) -> int | None:
        """
        Helper method to get column index from an expression.
        If expression is None, None will be returned as column index.
        """
        ...


class PlannedMinutesRenderer(MinutesRendererBase):
    """
    Custom renderer to allow access to planned minutes for planning list
    and pivot tables in resource planning.
    """

    def get_value(self, rowobj: object, expression: str | None, subscriber: object) -> int | None:
        """
        Standard method to get render planned minutes for a
        column index and a row object.
        If the column index is None, it will render the
        aggregated planned minutes for the row object.
        """
        ...

    def set_value(self, rowobj: object, expression: str | None, value: int | None) -> None:
        """
        Standard method to get set planned minutes for a
        column index and a row object.
        If the column index is None, setting planned minutes
        is not allowed and an exception will be thrown.
        """
        ...

    def get_is_readonly(self, rowobj: object, expression: str | None, subscriber: None) -> bool:
        """
        Standard method to evalualte, if planned minutes can be set
        for a column index and a row object.
        If the column index is None, setting planned minutes
        is not allowed.
        """
        ...


class NetCapacityRenderer(MinutesRendererBase):
    """
    Custom renderer to calculate the net capacity for a worker.
    If renderer is placed in a dynamic column, the capacity will only be
    calculated for that column and worker, otherwhise it will
    calculate the capacity for all columns and that worker.
    This renderer can only be used in worker lists.
    """

    is_readonly: bool = ...
    """
    Capacity renderer are all readonly.
    Standard attribute to indicate that renderer is readonly.
    """

    def get_value(self, rowobj: object, expression: str | None, subscriber: object) -> int:
        """
        Standard method to render the net capacity in minutes.
        """
        ...


class GrossCapacityRenderer(MinutesRendererBase):
    """
    Custom renderer to calculate the gross capacity for a worker.
    If renderer is placed in a dynamic column, the capacity will only be
    calculated for that column and worker, otherwhise it will
    calculate the capacity for all columns and that worker.
    This renderer can only be used in worker lists.
    """

    is_readonly: bool = ...
    """
    Capacity renderer are all readonly.
    Standard attribute to indicate that renderer is readonly.
    """

    def get_value(self, rowobj: object, expression: str | None, subscriber: object) -> int:
        """
        Standard method to render the gross capacity in minutes.
        """
        ...


class RemainingCapacityRenderer(MinutesRendererBase):
    """
    Custom renderer to calculate the remaining capacity for a worker.
    If renderer is placed in a dynamic column, the capacity will only be
    calculated for that column and worker, otherwhise it will
    calculate the capacity for all columns and that worker.
    This renderer can only be used in worker lists.
    """

    is_readonly: bool = ...
    """
    Capacity renderer are all readonly.
    Standard attribute to indicate that renderer is readonly.
    """

    def get_value(self, rowobj: object, expression: str | None, subscriber: object) -> int:
        """
        Standard method to render the remaining capacity in minutes.
        """
        column_index = self.get_column_index_from_expression(expression)
        return self.get_controller().get_remaining_capacity(column_index, rowobj, subscriber)

    def get_text_color(self, rowobj, expression, subscriber):
        """
        Standard method which is called to render the text color for a cell.
        Returns None or  text color as string.
        """
        value = self.get_value(rowobj, expression, subscriber)

        if value is not None and value < 0:
            return "clDarkRed"

        return None


class ShowBulkPlanningDialogRenderer(RendererBase):
    """
    Custom renderer for the button to show a bulk planning dialog.
    """

    display_type: str = ...
    """
    Display type shall be button, that a button will be shown in the planning table.
    Standard attribute set a specific display type.
    """

    def _is_visible(self) -> bool:
        """
        Private helper function to check, if planning table is writable or not
        """
        ...

    def get_buttoniconid(self, rowobj: object, expression: str | None, subscriber: object) -> str:
        """
        Standard method to get button icon id as string.
        If no button icon should be shown, an empty string will be returned.
        """
        ...

    def get_buttontooltip(self, rowobj: object, expression: str | None, subscriber: object) -> str:
        """
        Standard method to get button tooltip text as string.
        If no button tooltip should be shown, an empty string will be returned.
        """
        ...

    def button_clicked(self, rowobj: object, expression: str | None) -> None:
        """
        Standard method to show the bulk planning dialog when the button
        has been clicked.
        """
        ...
